'use client';

import { motion, useScroll, useTransform } from 'framer-motion';
import Image from 'next/image';
import Link from 'next/link';
import { useRef } from 'react';

const Hero = () => {
    const ref = useRef<HTMLDivElement>(null);
    const { scrollYProgress } = useScroll({ target: ref, offset: ["start end", "end start"] });

    // Efek parallax: teks bergerak ke atas 50% dari tinggi section saat discroll.
    const y = useTransform(scrollYProgress, [0, 1], ["0%", "50%"]);

    // Variants untuk animasi teks dan tombol saat masuk halaman.
    const textVariants = {
        hidden: { opacity: 0, y: 50 }, // Mulai dari transparan, 50px di bawah
        visible: {
            opacity: 1, // Menjadi penuh opacity
            y: 0,       // Kembali ke posisi awal
            transition: { duration: 0.9, ease: "easeInOut" } // Menggunakan easing standar
        }
    };

    const buttonVariants = {
        hidden: { opacity: 0, scale: 0.8 }, // Mulai dari transparan, ukuran 80%
        visible: {
            opacity: 1, // Menjadi penuh opacity
            scale: 1,   // Kembali ke ukuran normal
            transition: { duration: 0.8, ease: "easeOut" } // Menggunakan easing standar
        }
    };

    return (
        <section
            ref={ref} // Referensi untuk efek parallax
            id="hero"
            className="relative h-screen overflow-hidden flex flex-col justify-center items-center text-bali-text-light text-center px-4 md:px-24"
        >
            {/* Background Image: Dioptimalkan dengan next/image */}
            <Image
                src="/images/hero-bg.jpg"
                alt="Beautiful Bali Landscape with rice fields and mountains"
                fill // Mengisi seluruh parent
                priority // Penting untuk gambar LCP (Largest Contentful Paint)
                className="object-cover -z-10" // Gambar menutupi area, di belakang konten
                sizes="(max-width: 768px) 100vw, 100vw" // Ukuran gambar responsif
            />

            {/* Overlay: untuk meningkatkan keterbacaan teks */}
            <div className="absolute inset-0 bg-gradient-to-t from-bali-primary/80 to-bali-primary/50 -z-10"></div>

            {/* Konten Hero: Diterapkan efek parallax `y` di sini */}
            <motion.div style={{ y }} className="relative z-10 flex flex-col items-center">
                <motion.p
                    className="text-xl md:text-2xl mb-4 font-body font-medium uppercase tracking-widest"
                    variants={textVariants}
                    initial="hidden"
                    animate="visible"
                    transition={{ delay: 0.5, ...textVariants.visible.transition }} // Delay 0.5s
                >
                    Your Journey to Paradise Awaits
                </motion.p>
                <motion.h1
                    className="text-5xl sm:text-6xl md:text-8xl font-bold leading-tight max-w-5xl font-heading mb-8"
                    variants={textVariants}
                    initial="hidden"
                    animate="visible"
                    transition={{ delay: 0.7, ...textVariants.visible.transition }} // Delay 0.7s
                >
                    Explore The <span className="text-bali-accent">Mystical Beauty</span> of Bali
                </motion.h1>
                <motion.a
                    href="#about"
                    // Fungsi onClick untuk smooth scroll manual
                    onClick={(e) => { e.preventDefault(); document.getElementById('about')?.scrollIntoView({ behavior: 'smooth' }); }}
                    className="btn-primary mt-8 text-xl" // Menggunakan kelas button kustom
                    variants={buttonVariants}
                    initial="hidden"
                    animate="visible"
                    transition={{ delay: 1, ...buttonVariants.visible.transition }} // Delay 1s
                >
                    Start Your Adventure
                </motion.a>
            </motion.div>
        </section>
    );
};

export default Hero;