'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { motion, AnimatePresence } from 'framer-motion';

const Header = () => {
    const [isMenuOpen, setIsMenuOpen] = useState(false);
    const [isScrolled, setIsScrolled] = useState(false);

    useEffect(() => {
        const handleScroll = () => {
            // Mengubah state `isScrolled` berdasarkan posisi scroll
            setIsScrolled(window.scrollY > 80); // Mengaktifkan efek setelah scroll 80px
        };

        // Menambahkan event listener saat komponen mount
        window.addEventListener('scroll', handleScroll);
        // Membersihkan event listener saat komponen unmount
        return () => {
            window.removeEventListener('scroll', handleScroll);
        };
    }, []); // [] agar efek hanya dijalankan sekali setelah mount

    // Fungsi untuk melakukan smooth scroll ke section tertentu
    const scrollToSection = (id: string) => {
        const element = document.getElementById(id);
        if (element) {
            const headerOffset = 90; // Offset untuk header tetap (fixed header)
            const elementPosition = element.getBoundingClientRect().top + window.scrollY;
            const offsetPosition = elementPosition - headerOffset;

            window.scrollTo({
                top: offsetPosition,
                behavior: 'smooth'
            });
            setIsMenuOpen(false); // Tutup menu mobile setelah klik
        }
    };

    // Variants Framer Motion untuk animasi menu mobile
    const menuVariants = {
        hidden: { opacity: 0, x: "100%" }, // Mulai dari luar layar kanan
        visible: { opacity: 1, x: 0, transition: { type: "tween", duration: 0.3 } }, // Meluncur masuk
        exit: { opacity: 0, x: "100%", transition: { type: "tween", duration: 0.3 } } // Meluncur keluar
    };

    const navItemVariants = {
        hidden: { opacity: 0, y: 20 }, // Mulai dari transparan dan sedikit ke bawah
        visible: { opacity: 1, y: 0, transition: { duration: 0.3 } } // Meluncur masuk
    };

    return (
        <motion.header
            // Animasi saat header muncul pertama kali
            initial={{ y: -100 }}
            animate={{ y: 0 }}
            transition={{ type: "spring", stiffness: 120, damping: 15 }} // Animasi pegas
            // Kelas Tailwind CSS untuk styling responsif dan transisi background/shadow
            className={`fixed top-0 left-0 w-full flex justify-between items-center py-5 px-10 md:px-24 z-50 transition-all duration-300
        ${isScrolled ? 'bg-bali-primary shadow-xl py-4' : 'bg-transparent py-5'}`} // Efek scroll: ubah background dan padding
        >
            {/* Logo Website */}
            <Link href="/" className="logo text-bali-text-light text-3xl font-bold font-heading hover:text-bali-accent transition-colors">
                Bali.ID
            </Link>

            {/* Desktop Navigation */}
            <nav className="hidden md:flex items-center gap-10">
                {/* Menggunakan tag <a> dan onClick untuk smooth scroll, dengan href sebagai fallback dan semantic */}
                <a
                    href="#about"
                    onClick={(e) => { e.preventDefault(); scrollToSection('about'); }}
                    className="text-bali-text-light text-lg font-body hover:text-bali-accent transition-colors hover:scale-105 transform origin-center"
                >
                    About
                </a>
                <a
                    href="#discover"
                    onClick={(e) => { e.preventDefault(); scrollToSection('discover'); }}
                    className="text-bali-text-light text-lg font-body hover:text-bali-accent transition-colors hover:scale-105 transform origin-center"
                >
                    Discover
                </a>
                <a
                    href="#testimonials"
                    onClick={(e) => { e.preventDefault(); scrollToSection('testimonials'); }}
                    className="text-bali-text-light text-lg font-body hover:text-bali-accent transition-colors hover:scale-105 transform origin-center"
                >
                    Testimonials
                </a>
                <a
                    href="#contact"
                    onClick={(e) => { e.preventDefault(); scrollToSection('contact'); }}
                    className="btn-secondary ml-4 text-sm px-6 py-2" // Menggunakan button style yang baru dari globals.css
                >
                    Contact Us
                </a>
            </nav>

            {/* Mobile Menu Icon (Hamburger) */}
            <div className="md:hidden cursor-pointer text-bali-text-light text-3xl" onClick={() => setIsMenuOpen(!isMenuOpen)}>
                {/* Icon hamburger SVG yang lebih modern dari Heroicons atau serupa */}
                <svg xmlns="http://www.w3.org/2000/svg" className="h-7 w-7" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2}>
                    <path strokeLinecap="round" strokeLinejoin="round" d="M4 6h16M4 12h16M4 18h16" />
                </svg>
            </div>

            {/* Mobile Menu Overlay */}
            <AnimatePresence>
                {isMenuOpen && (
                    <motion.div
                        className="fixed inset-0 bg-bali-primary bg-opacity-95 flex flex-col items-center justify-center space-y-8 md:hidden shadow-lg z-50"
                        variants={menuVariants} // Menerapkan varian animasi
                        initial="hidden"        // Mulai dari hidden state
                        animate="visible"       // Animasikan ke visible state
                        exit="exit"             // Animasikan saat keluar (saat `isMenuOpen` false)
                    >
                        {/* Tombol tutup menu mobile */}
                        <motion.button
                            className="absolute top-6 right-8 text-bali-text-light text-5xl"
                            onClick={() => setIsMenuOpen(false)}
                        >
                            &times;
                        </motion.button>
                        {/* Item navigasi mobile */}
                        <motion.div variants={navItemVariants}>
                            <a
                                href="#about"
                                onClick={(e) => { e.preventDefault(); scrollToSection('about'); }}
                                className="text-bali-text-light text-3xl font-heading font-semibold"
                            >
                                About
                            </a>
                        </motion.div>
                        <motion.div variants={navItemVariants}>
                            <a
                                href="#discover"
                                onClick={(e) => { e.preventDefault(); scrollToSection('discover'); }}
                                className="text-bali-text-light text-3xl font-heading font-semibold"
                            >
                                Discover
                            </a>
                        </motion.div>
                        <motion.div variants={navItemVariants}>
                            <a
                                href="#testimonials"
                                onClick={(e) => { e.preventDefault(); scrollToSection('testimonials'); }}
                                className="text-bali-text-light text-3xl font-heading font-semibold"
                            >
                                Testimonials
                            </a>
                        </motion.div>
                        <motion.div variants={navItemVariants}>
                            <a
                                href="#contact"
                                onClick={(e) => { e.preventDefault(); scrollToSection('contact'); }}
                                className="text-bali-text-light text-3xl font-heading font-semibold"
                            >
                                Contact
                            </a>
                        </motion.div>
                    </motion.div>
                )}
            </AnimatePresence>
        </motion.header>
    );
};

export default Header;